<?php
// ---------- Helpers ----------
/**
 * Retourne le timestamp (Unix, UTC) de la prochaine occurrence de $time_str (format 'H:i')
 * dans le fuseau horaire du site (wp_timezone()).
 */
function sayens_get_next_timestamp_for_time( string $time_str ): int {
    $tz  = wp_timezone();
    $now = new DateTimeImmutable('now', $tz);

    $target = DateTimeImmutable::createFromFormat('Y-m-d H:i', $now->format('Y-m-d') . ' ' . $time_str, $tz);
    if (! $target) {
        // fallback: tomorrow same heure
        return $now->modify('+1 day')->getTimestamp();
    }

    // si la cible est passée aujourd'hui, alors on créé celle de demain
    if ($target->getTimestamp() <= $now->getTimestamp()) {
        $target = $target->modify('+1 day');
    }

    // getTimestamp() renvoie un Unix timestamp (UTC), compatible avec wp_schedule_*
    return $target->getTimestamp();
}

// ---------- Activation / Désactivation ----------
add_action('after_setup_theme', function () {
    $times = ['07:30', '11:30', '15:30'];

    foreach ($times as $t) {
        // évite les doublons : on utilise l'argument (time) pour différencier
        if (! wp_next_scheduled('sayens_export_at_time', array($t)) ) {
            $ts = sayens_get_next_timestamp_for_time($t);
            wp_schedule_single_event($ts, 'sayens_export_at_time', array($t));
        }
    }

    // Option utile : initialiser la dernière exécution à maintenant pour ne pas exporter l'historique
    if ( get_option('sayens_last_export_time_gmt') === false ) {
        update_option('sayens_last_export_time_gmt', time());
    }
});

add_action('switch_theme', function () {
    // supprime tous les single events attachés au hook
    wp_clear_scheduled_hook('sayens_export_at_time');
});

// ---------- Handler ----------
add_action('sayens_export_at_time', 'sayens_export_at_time_handler', 10, 1);
function sayens_export_at_time_handler( $time_str = '07:30' ) {
    // 1) Re-scheduler la prochaine occurrence (jour suivant)
    $next_ts = sayens_get_next_timestamp_for_time($time_str);
    if (! wp_next_scheduled('sayens_export_at_time', array($time_str)) ) {
        wp_schedule_single_event($next_ts, 'sayens_export_at_time', array($time_str));
    }

    // 2) Récupérer la période à exporter (en GMT)
    $last_run_gmt = (int) get_option('sayens_last_export_time_gmt', 0);
    $now_gmt      = time(); // UTC timestamp

    //error_log("[Sayens] déclenché pour {$time_str}. last_run_gmt={$last_run_gmt} now_gmt={$now_gmt}");

    // 3) Récupérer commandes crées STRICTEMENT après last_run_gmt (post_date_gmt)
    $after = ($last_run_gmt > 0) ? gmdate('Y-m-d H:i:s', $last_run_gmt) : '1970-01-01 00:00:00';

    $order_ids = wc_get_orders(array(
        'status'     => array('wc-processing', 'wc-completed'), // adapte si besoin
        'limit'      => -1,
        'return'     => 'ids',
        'orderby'    => 'date',
        'order'      => 'ASC',
        'date_query' => array(
            array(
                'after'     => $after,
                'column'    => 'post_date_gmt',
                'inclusive' => false,
            ),
        ),
    ));

    //error_log('[Sayens] commandes trouvées : ' . count($order_ids));

    // 4) Exporter (ici on appelle ta fonction existante)
    foreach ($order_ids as $oid) {
        // ta fonction devrait accepter l'ID et faire l'export
        export_order_to_xml($oid);
    }

    // 5) Mettre à jour la dernière exécution (en GMT)
    update_option('sayens_last_export_time_gmt', $now_gmt);
}

/* add_action('admin_init', function() {
    if ( current_user_can('manage_woocommerce') && isset($_GET['sayens_test']) ) {
        // Ex: ?sayens_test=11:30
        $time = sanitize_text_field($_GET['sayens_test']);
        do_action('sayens_export_at_time', $time);
        wp_die('Test Sayens effectué pour '.$time);
    }
}); */