<?php
require_once get_stylesheet_directory() . '/vendor/autoload.php';

use libphonenumber\PhoneNumberUtil;
use libphonenumber\PhoneNumberFormat;


/*
 * Création d'un fichier XML à chaque commande de WooCommerce pour SAYENS
 */

function export_order_to_xml($order_id) {
    // Vérification si l'ID de commande est valide
    if (!$order_id) return;

    // Charger la commande
    $order = wc_get_order($order_id);
    if (!$order) return;

    $order_id = $order->get_id();

    if($order->get_total() <= 0){
        return;
    }

    
    // Création du document XML
    $xml = new DOMDocument('1.0', 'UTF-8');
    $xml->formatOutput = true;

    // Élément racine
    $commandes = $xml->createElement('commandes');
    $xml->appendChild($commandes);

    // Élément commande
    $commande = $xml->createElement('commande');
    $commande->setAttribute('num_cde', $order_id);
    $commandes->appendChild($commande);

    $company_name = !empty($order->get_billing_company()) ? $order->get_billing_company() : $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();

    // Date de commande
    $commande->appendChild($xml->createElement('date_commande', $order->get_date_created()->date('d/m/Y')));

    // Objet commande
    $commande->appendChild($xml->createElement('objet_commande', 'Commande ' . $company_name . ' n° ' . $order_id));

    // Montant
    $commande->appendChild($xml->createElement('montant_cde', number_format($order->get_total(), 2, ',', '')));

    // Récupère le moyen de paiement utilisé
    $payment_method = $order->get_payment_method();
    
    // Vérifie si le paiement a été effectué par carte
    $is_paid_by_card = in_array($payment_method, ['payplug'], true);

    // Définit la valeur de <payer>
    $payer_value = $is_paid_by_card ? 'Oui' : 'Non';

    // Créer le noeud <payer> dans le XML
    $commande->appendChild($xml->createElement('payer', $payer_value));

    // Numéro facture proforma
    $commande->appendChild($xml->createElement('num_facture_proforma', 'FAC-' . $order_id));

    // Élément client
    $client = $xml->createElement('client');
    $commande->appendChild($client);

    $siret = !empty($order->get_meta('_billing_siret')) ? $order->get_meta('_billing_siret') : '';
    if(!empty($siret)){
        // récupère les 9 premiers chiffres du SIRET pour faire le SIREN
        $siren = substr($siret, 0, 9);
    }
    $tva_intra = !empty($order->get_meta('_billing_taxe_intra')) ? $order->get_meta('_billing_taxe_intra') : '';
    $type_activite = !empty($order->get_meta('_billing_type_activite')) ? $order->get_meta('_billing_type_activite') : '';

    // Type de client
    if(!empty($siret) && !empty($company_name)){
        $is_company = 'S';
    }else{
        $is_company = 'P';
    }

    $client->appendChild($xml->createElement('type_client', $is_company));

    // Informations client
    $client->appendChild($xml->createElement('nom_client', $order->get_billing_first_name() . ' ' . $order->get_billing_last_name()));
    $client->appendChild($xml->createElement('mail_compte', $order->get_billing_email()));

    // Adresse de facturation
    $country_code = $order->get_billing_country();
    $countries = WC()->countries->get_countries();
    $country_name = isset($countries[$country_code]) ? $countries[$country_code] : $country_code;

    $adresse_facturation = $xml->createElement('adresse_facturation');
    $client->appendChild($adresse_facturation);
    $adresse_facturation->appendChild($xml->createElement('nom', $company_name));
    $adresse_facturation->appendChild($xml->createElement('nature', $type_activite));
    $adresse_facturation->appendChild($xml->createElement('adresse_1', $order->get_billing_address_1()));
    $adresse_facturation->appendChild($xml->createElement('adresse_2', $order->get_billing_address_2()));
    $adresse_facturation->appendChild($xml->createElement('cp', $order->get_billing_postcode()));
    $adresse_facturation->appendChild($xml->createElement('ville', $order->get_billing_city()));
    $adresse_facturation->appendChild($xml->createElement('pays', $country_name));
    $adresse_facturation->appendChild($xml->createElement('mail', $order->get_billing_email()));

    // Récupère le téléphone et le passe par la librairie libphonenumber for PHP pour le convertir en numéro avec le préfixe de pays
    $phone = $order->get_billing_phone();
    $phone_number_util = PhoneNumberUtil::getInstance();

    try {
        $parsed_number = $phone_number_util->parse($phone, $country_code);

        // Format E164
        $formatted_number = $phone_number_util->format($parsed_number, PhoneNumberFormat::E164);

        // Vérifier si le numéro est mobile
        $is_mobile = $phone_number_util->getNumberType($parsed_number) === \libphonenumber\PhoneNumberType::MOBILE;

        // Assignation au fixe ou mobile
        $fixed_phone = !$is_mobile ? $formatted_number : '';
        $mobile_phone = $is_mobile ? $formatted_number : '';

    } catch (\libphonenumber\NumberParseException $e) {
        // En cas d'erreur, log et utiliser les valeurs brutes
        //error_log('Erreur dans le formatage du numéro : ' . $e->getMessage());
        $fixed_phone = '';
        $mobile_phone = $phone; // Mettre le numéro brut si l'analyse échoue
    }
    
    // Créer les noeuds pour le numéro de téléphone (fixe et mobile) formatté
    $adresse_facturation->appendChild($xml->createElement('telephone', $fixed_phone ?: ''));
    $adresse_facturation->appendChild($xml->createElement('portable', $mobile_phone ?: ''));

    
    // Si c'est une entreprise et en France, on doit ajouter le SIREN, SIRET et le Numéro de TVA Intracommunautaire
    if(($is_company == 'S') && ($order->get_billing_country() == 'FR')){
        $adresse_facturation->appendChild($xml->createElement('siren', $siren ?: ''));
        $adresse_facturation->appendChild($xml->createElement('siret', $siret ?: ''));
    }

    if(!empty($tva_intra)){
        $adresse_facturation->appendChild($xml->createElement('tva_intra', $tva_intra));
    }


    // Vérifier si l'adresse de livraison diffère
    $is_shipping_different = $order->get_shipping_address_1() && (
        $order->get_shipping_address_1() !== $order->get_billing_address_1() ||
        $order->get_shipping_postcode() !== $order->get_billing_postcode() ||
        $order->get_shipping_city() !== $order->get_billing_city()
    );

    // Ajouter l'élément <adresse_livraison>
    $adresse_livraison = $xml->createElement('adresse_livraison');
    $client->appendChild($adresse_livraison);

    // Indiquer si l'adresse est différente
    $adresse_livraison->appendChild($xml->createElement('adresse_livraison_differente', $is_shipping_different ? 'Oui' : 'Non'));

    if ($is_shipping_different) {
        $adresse_livraison->appendChild($xml->createElement('nom', $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name()));
        if($is_company == 'S'){
            $adresse_livraison->appendChild($xml->createElement('siret', $order->get_meta('billing_siret'))); // Placeholder si requis
        }
        $adresse_livraison->appendChild($xml->createElement('adresse_1', $order->get_shipping_address_1()));
        $adresse_livraison->appendChild($xml->createElement('adresse_2', $order->get_shipping_address_2()));
        $adresse_livraison->appendChild($xml->createElement('cp', $order->get_shipping_postcode()));
        $adresse_livraison->appendChild($xml->createElement('ville', $order->get_shipping_city()));

        $country_code = $order->get_shipping_country();
        $countries = WC()->countries->get_countries();
        $country_name = isset($countries[$country_code]) ? $countries[$country_code] : $country_code;
        $adresse_livraison->appendChild($xml->createElement('pays', $country_name)); // Code pays
        
        // Récupère le téléphone et le passe par la librairie libphonenumber for PHP pour le convertir en numéro avec le préfixe de pays
        $shipping_phone = $order->get_shipping_phone();

        try {
            $parsed_number = $phone_number_util->parse($shipping_phone, $country_code);

            // Format international
            $formatted_number = $phone_number_util->format($parsed_number, PhoneNumberFormat::INTERNATIONAL);

            // Vérifier si le numéro est mobile
            $is_mobile = $phone_number_util->getNumberType($parsed_number) === \libphonenumber\PhoneNumberType::MOBILE;

            // Assignation au fixe ou mobile
            $fixed_phone = !$is_mobile ? $formatted_number : '';
            $mobile_phone = $is_mobile ? $formatted_number : '';

        } catch (\libphonenumber\NumberParseException $e) {
            // En cas d'erreur, log et utiliser les valeurs brutes
            //error_log('Erreur dans le formatage du numéro : ' . $e->getMessage());
            $fixed_phone = '';
            $mobile_phone = $phone; // Mettre le numéro brut si l'analyse échoue
        }

        // Créer les noeuds pour le numéro de téléphone (fixe et mobile) formatté
        $adresse_livraison->appendChild($xml->createElement('telephone', $fixed_phone ?: ''));
        $adresse_livraison->appendChild($xml->createElement('portable', $mobile_phone ?: ''));

        $adresse_livraison->appendChild($xml->createElement('mail', $order->get_billing_email())); // Peut être réutilisé
    }

    // Ajouter l'élément <adresse_livraison>
    $adresse_siege = $xml->createElement('adresse_siege');
    $client->appendChild($adresse_siege);

    $adresse_siege->appendChild($xml->createElement('adresse_siege_differente', 'Non'));

    $order = wc_get_order($order_id);

    
    // Produits de la commande
    foreach ($order->get_items() as $item_id => $item) {
        $ligne_commande = $xml->createElement('ligne_commande');
        $ligne_commande->setAttribute('numero', $item_id);

        $ligne_commande->appendChild($xml->createElement('code_produit', $item->get_product_id()));
        $ligne_commande->appendChild($xml->createElement('designation', $item->get_name()));
        $ligne_commande->appendChild($xml->createElement('quantite', number_format($item->get_quantity(), 2, ',', '')));

        // On récupère le montant de TVA du produit
        $tax_amount = $item->get_total_tax();

        // Si c'est (la France ou Monaco) ou (un pays de l'UE et particulier) on met le montant de TVA à 0
        if((is_country_in_eu($country_code) && ($is_company == 'P')) || !tva_to_zero($country_code)){
            $tax_amount = 0;
        }

        // Si c'est un autre pays que la France et Monaco, le prix unitaire est le prix TTC, sinon c'est le HT
        $ligne_commande->appendChild($xml->createElement('prix_unitaire', number_format(($item->get_total() + $tax_amount) / $item->get_quantity(), 2, ',', '')));

        $product = $item->get_product();

        if ( $product->is_taxable() ) {        
            $tax_rates = WC_Tax::get_rates( $product->get_tax_class() );

            // Récupère le taux de TVA pour le produit        
            $tax_rate = array();
            
            // S'il y une TVA standard appliquée au produit, on la récupère avec get_rates_for_tax_class()
            if(empty( $tax_rates)){
                if(tva_to_zero($country_code)){
                    $tax_rate = get_default_tax_rate();
                }else{
                    $tax_rate['rate'] = 0;
                }
            }else{
                // Si c'est la France, Monaco ou un particulier en Europe, on met la TVA
                if(!tva_to_zero($country_code) || (is_country_in_eu($country_code) && ($is_company == 'P'))){
                    $tax_rate = reset( $tax_rates );
                }else{
                    $tax_rate['rate'] = 0;
                }
            }
            
            $tax_percentage = isset( $tax_rate['rate'] ) ? floatval($tax_rate['rate']) : 0;
        }        
        
        $ligne_commande->appendChild($xml->createElement('taux_tva', $tax_percentage));
        
        $commande->appendChild($ligne_commande);
    }


    // Ajoute une ligne commande s'il y a des frais de port
    if(!empty($order->get_shipping_total())){
        $ligne_commande = $xml->createElement('ligne_commande');
        $shipping_tax_percentage = 0;

        // Si c'est (la France ou Monaco) ou (un pays de l'UE et particulier) on met le montant de TVA à 20%
        if((is_country_in_eu($country_code) && ($is_company == 'P')) || !tva_to_zero($country_code)){
            $shipping_tax_percentage = 20;
        }
    
        $ligne_commande->appendChild($xml->createElement('code_produit', 1));
        $ligne_commande->appendChild($xml->createElement('designation', 'Frais de port'));
        $ligne_commande->appendChild($xml->createElement('quantite', number_format(1, 2, ',', '')));
        $ligne_commande->appendChild($xml->createElement('prix_unitaire', number_format($order->get_shipping_total(), 2, ',', '')));
        $ligne_commande->appendChild($xml->createElement('taux_tva', $shipping_tax_percentage));
        $commande->appendChild($ligne_commande);
    }


    // Sauvegarde du fichier XML
    $upload_dir = ABSPATH . 'sayens';
    if (!file_exists($upload_dir)) mkdir($upload_dir, 0755, true);

    // Si la commande a été payée avec un bon de commande, on le copie, le renomme avec le $order_id et le place dans le dossier avec les xml
    $bon_commande = get_post_meta( $order_id, 'ocim_bdc', true );
    if ( !empty($bon_commande)) {
        $bon_url = unserialize(urldecode($bon_commande))["file"];

        if(!empty($bon_url)){
            // Copie le fichier, le renomme avec l'id de la commande et le colle dans le dossier Sayens
            $file_content = file_get_contents($bon_url);
            if ($file_content !== false) {
                $file_extension = pathinfo($bon_url, PATHINFO_EXTENSION);
                $new_filename = $order_id . '_BDC.' . $file_extension;
                $new_file_path = $upload_dir . '/' . $new_filename;
                file_put_contents($new_file_path, $file_content);
            }
        }
    }


    // Si il y a un justificatif dans la commande, on le copie, le renomme avec le $order_id et le place dans le dossier avec les xml
    $justif = get_post_meta( $order_id, 'ocim_justif', true );
    if ( !empty($justif)) {
        $justif_url = unserialize(urldecode($justif))["file"];

        if(!empty($justif_url)){
            // Copie le fichier, le renomme avec l'id de la commande et le colle dans le dossier Sayens
            $file_content = file_get_contents($justif_url);
            if ($file_content !== false) {
                $file_extension = pathinfo($justif_url, PATHINFO_EXTENSION);
                $new_filename = $order_id . '_JUSTIF.' . $file_extension;
                $new_file_path = $upload_dir . '/' . $new_filename;
                file_put_contents($new_file_path, $file_content);
            }
        }
    }

    $file_path = $upload_dir . '/' . $order_id . '_CDE.xml';
    $xml->save($file_path);
}
